package de.thedevstack.android.nextcloud.bookmark.share.activities;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.annotation.TargetApi;
import android.content.Intent;
import android.content.SharedPreferences;
import android.preference.PreferenceManager;
import android.support.v7.app.AppCompatActivity;

import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.KeyEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.inputmethod.EditorInfo;
import android.webkit.URLUtil;
import android.widget.EditText;
import android.widget.TextView;

import de.thedevstack.android.nextcloud.bookmark.share.Constants;
import de.thedevstack.android.nextcloud.bookmark.share.NextcloudBookmark;
import de.thedevstack.android.nextcloud.bookmark.share.R;
import de.thedevstack.android.nextcloud.bookmark.share.async.NextcloudCheckCredentialsTask;

/**
 * A login screen that offers login via username/password.
 */
public class NextcloudPreferencesActivity extends AppCompatActivity {
    /**
     * Keep track of the login task to ensure we can cancel it if requested.
     */
    private NextcloudCheckCredentialsTask checkCredentialsTask = null;

    // UI references.
    private EditText serverUrlView;
    private EditText usernameView;
    private EditText passwordView;
    private View mProgressView;
    private View mLoginFormView;
    private NextcloudBookmark bookmark;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_nextcloud_preferences);

        Intent intent = getIntent();

        this.bookmark = NextcloudBookmark.create(intent.getStringExtra(Intent.EXTRA_TEXT), intent.getStringExtra(Intent.EXTRA_TITLE));

        SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(getApplicationContext());

        // Set up the login form.
        serverUrlView = findViewById(R.id.serverUrl);
        serverUrlView.setText(prefs.getString(Constants.SERVER_URL_PREFERENCE_KEY, getString(R.string.default_server_url)));
        usernameView = findViewById(R.id.username);
        usernameView.setText(prefs.getString(Constants.USERNAME_PREFERENCE_KEY, getString(R.string.default_blank)));

        passwordView = findViewById(R.id.password);
        passwordView.setText(prefs.getString(Constants.PASSWORD_PREFERENCE_KEY, getString(R.string.default_blank)));
        passwordView.setOnEditorActionListener(new TextView.OnEditorActionListener() {
            @Override
            public boolean onEditorAction(TextView textView, int id, KeyEvent keyEvent) {
                if (id == EditorInfo.IME_ACTION_DONE || id == EditorInfo.IME_NULL) {
                    checkCredentials();
                    return true;
                }
                return false;
            }
        });

        findViewById(R.id.sign_in_button).setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                checkCredentials();
            }
        });

        mLoginFormView = findViewById(R.id.login_form);
        mProgressView = findViewById(R.id.login_progress);
    }

    /**
     * Attempts to sign in or register the account specified by the login form.
     * If there are form errors (invalid email, missing fields, etc.), the
     * errors are presented and no actual login attempt is made.
     */
    private void checkCredentials() {
        if (checkCredentialsTask != null) {
            return;
        }

        // Reset errors.
        serverUrlView.setError(null);
        usernameView.setError(null);
        passwordView.setError(null);

        // Store values at the time of the login attempt.
        String nextcloudServerUrl = serverUrlView.getText().toString();
        String username = usernameView.getText().toString();
        String password = passwordView.getText().toString();

        boolean cancel = false;
        View focusView = null;

        if (TextUtils.isEmpty(nextcloudServerUrl) || !URLUtil.isNetworkUrl(nextcloudServerUrl)) {
            serverUrlView.setError(getString(R.string.error_server_url_invalid));
            focusView = serverUrlView;
            cancel = true;
        }

        // Check for a valid email address.
        if (TextUtils.isEmpty(username)) {
            usernameView.setError(getString(R.string.error_field_required));
            focusView = usernameView;
            cancel = true;
        }

        if (cancel) {
            // There was an error; don't attempt login and focus the first
            // form field with an error.
            focusView.requestFocus();
        } else {
            // Show a progress spinner, and kick off a background task to
            // perform the user login attempt.
            showProgress(true);
            checkCredentialsTask = new NextcloudCheckCredentialsTask(this, this.bookmark, nextcloudServerUrl, username, password);
            checkCredentialsTask.execute((Intent) null);
        }
    }

    /**
     * Shows the progress UI and hides the login form.
     */
    @TargetApi(Build.VERSION_CODES.HONEYCOMB_MR2)
    public void showProgress(final boolean show) {
        // On Honeycomb MR2 we have the ViewPropertyAnimator APIs, which allow
        // for very easy animations. If available, use these APIs to fade-in
        // the progress spinner.
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB_MR2) {
            int shortAnimTime = getResources().getInteger(android.R.integer.config_shortAnimTime);

            mLoginFormView.setVisibility(show ? View.GONE : View.VISIBLE);
            mLoginFormView.animate().setDuration(shortAnimTime).alpha(
                    show ? 0 : 1).setListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationEnd(Animator animation) {
                    mLoginFormView.setVisibility(show ? View.GONE : View.VISIBLE);
                }
            });

            mProgressView.setVisibility(show ? View.VISIBLE : View.GONE);
            mProgressView.animate().setDuration(shortAnimTime).alpha(
                    show ? 1 : 0).setListener(new AnimatorListenerAdapter() {
                @Override
                public void onAnimationEnd(Animator animation) {
                    mProgressView.setVisibility(show ? View.VISIBLE : View.GONE);
                }
            });
        } else {
            // The ViewPropertyAnimator APIs are not available, so simply show
            // and hide the relevant UI components.
            mProgressView.setVisibility(show ? View.VISIBLE : View.GONE);
            mLoginFormView.setVisibility(show ? View.GONE : View.VISIBLE);
        }
    }

    public void stopProgress() {
        this.checkCredentialsTask = null;
        showProgress(false);
    }

    public void showIncorrectPassword() {
        passwordView.setError(getString(R.string.error_incorrect_password));
        passwordView.requestFocus();
    }
}

